using System;
using System.IO;
using DynaPDF;

namespace text_extraction3
{
	class TextExtraction3
	{
      /*
         Note that the dynapdf.dll must be copied into the output directory or into a
         Windwos search path (e.g. %WINDOWS%/System32) before the application can be executed!
      */

      static private void WritePageIdentifier(System.IO.BinaryWriter F, int PageNum)
      {
         if (PageNum > 1)
         {
            F.Write(System.Text.UnicodeEncoding.Unicode.GetBytes("\r\n"));
         }
         F.Write(System.Text.UnicodeEncoding.Unicode.GetBytes(String.Format("%----------------------- Page {0} -----------------------------\r\n", PageNum)));
      }

      // Error callback function.
      // If the function name should not appear at the beginning of the error message then set
      // the flag emNoFuncNames (pdf.SetErrorMode(DynaPDF.TErrMode.emNoFuncNames);).
      static int PDFError(IntPtr Data, int ErrCode, IntPtr ErrMessage, int ErrType)
      {
         Console.Write("{0}\n", System.Runtime.InteropServices.Marshal.PtrToStringAnsi(ErrMessage));
         return 0; // We try to continue if an error occurrs. Any other return value breaks processing.
      }

		[STAThread]
		static void Main(string[] args)
		{
		   try
		   {
            CPDF pdf = new CPDF();
            // Error messages and warnings are passed to the callback function.
            pdf.SetOnErrorProc(IntPtr.Zero, new DynaPDF.TErrorProc(PDFError));
            pdf.CreateNewPDF(null); // We create no PDF file in this example

            // External CMaps should always be loaded when processing text from PDF files.
            // See the description of GetPageText() for further information.
            pdf.SetCMapDir(System.IO.Path.GetFullPath("../../../../Resource/CMap/"), DynaPDF.TLoadCMapFlags.lcmDelayed | DynaPDF.TLoadCMapFlags.lcmRecursive);

            // We avoid the conversion of pages to templates
            pdf.SetImportFlags(DynaPDF.TImportFlags.ifImportAll | DynaPDF.TImportFlags.ifImportAsPage);
            if (pdf.OpenImportFile("../../../../dynapdf_help.pdf", DynaPDF.TPwdType.ptOpen, null) < 0)
            {
               Console.Write("Input file \"../../../../../dynapdf_help.pdf\" not found!\n");
               Console.Read();
               return;
            }
            pdf.ImportPDFFile(1, 1.0, 1.0);
            pdf.CloseImportFile();

            // We flatten markup annotations and form fields so that we can extract the text of these objects too.
            pdf.FlattenAnnots(TAnnotFlattenFlags.affMarkupAnnots);
            pdf.FlattenForm();

            // We write the output file into the current directory.
            String outFile = Directory.GetCurrentDirectory() + "\\out.txt";
            System.IO.FileStream   strm = new System.IO.FileStream(outFile, System.IO.FileMode.Create, System.IO.FileAccess.ReadWrite);
            System.IO.BinaryWriter file = new System.IO.BinaryWriter(strm, System.Text.Encoding.Unicode);
            // Write a Little Endian marker to the file (byte order mark)
            file.Write(System.Text.UnicodeEncoding.Unicode.GetBytes("\uFEFF"));

            String outText = String.Empty;
            int pageCount = pdf.GetPageCount();
            for (int i = 1; i <= pageCount; i++)
            {
               WritePageIdentifier(file, i);
               // It is not recommended to sort text on the y-axis since causes sometimes strange results.
               if (pdf.ExtractText(i, TTextExtractionFlags.tefDeleteOverlappingText | TTextExtractionFlags.tefSortTextX, ref outText))
                  file.Write(System.Text.UnicodeEncoding.Unicode.GetBytes(outText));
            }
            file.Close();
            // Tell the garbage collector that these objects can be released now
            file = null;
            strm = null;
            pdf  = null;

            Console.Write("Text successfully extracted to \"{0}\"!\n", outFile);
            System.Diagnostics.Process p = new System.Diagnostics.Process();
            p.StartInfo.FileName = outFile;
            p.Start();
         }catch(Exception e)
         {
            Console.Write(e.Message);
            Console.Read();
         }
		}
	}
}
